﻿#include  "StdAfx.h"

#include  "Command.hpp"
#include  <szRuntimeException.hpp>
#include  <boost/algorithm/string.hpp>
#include  <boost/tokenizer.hpp>

SZ_AN_BEG

struct CommandDefinition
{
  CommandType    commandType;
  CommandArgType commandArgType;
  szstring       commandString;
} comDefs[] =
{
  // 必ず展開コマンドが先頭に置かれる
  { DecompressCommand,  Multiple, SZL("/D") }, // 展開（引数はそれぞれ展開するアーカイブファイルパス）
  { CompressCommand,    Multiple, SZL("/C") }, // 圧縮（引数は圧縮するファイル／ディレクトリパス、アーカイブ名は UI でユーザーに問い合わせ）
  { CompressAsCommand,  Multiple, SZL("/P") }, // 圧縮（第一引数はアーカイブパス、その他の引数は圧縮するファイル／ディレクトリパス）

  { InputPathCommand,   Single,   SZL("/I") }, // 入力ファイルのルートパスを設定（引数はルートの絶対パス）
  { OutputPathCommand,  Single,   SZL("/O") }, // 出力（圧縮も展開も共通）のルートパスを設定（引数はルートの絶対パス）

  { VerboseCommand,     Single,   SZL("/V") }, // エラーメッセージの冗長表示モードを設定（引数は on または off）

  //{ CommandFile, Single,   SZL("/E") }, // 外部コマンドファイル（唯一の引数がコマンドファイルパス）

  { UndefinedCommand,    None,     SZL("") },   // N/A（配列の終端要素）
};

CommandDefinition * MatchCommand(const szstring &orgStr)
{
  const szstring str(boost::algorithm::to_upper_copy(orgStr));

  CommandDefinition *def = comDefs;
  while (def->commandType != UndefinedCommand)
  {
    if (0 == str.compare(def->commandString))
      return def;
    ++def;
  }
  return 0;
}

SZ_AN_END

Command::Command(CommandType commandType, const std::vector<szstring> &commandArgs) : type(commandType), args(commandArgs)
{
}

Command::~Command()
{
}

std::deque<Command> ParseCommands(const szstring &cmdline)
{
  using namespace std;
  using namespace boost;
  using namespace szpp;

  deque<Command> commands;

  typedef escaped_list_separator<szchar> sep_type;
  typedef tokenizer< sep_type, szstring::const_iterator, szstring > tok_type;

  sep_type sep(SZL_EMPTY, SZL(" "), SZL("'\""));
  tok_type tok(cmdline, sep);

  CommandDefinition *curDef = 0;
  vector<szstring> currentArgs;

  for (tok_type::const_iterator it = tok.begin(); it != tok.end(); ++it)
  {
    CommandDefinition *def = MatchCommand(*it);
    if (def != 0)
    {
      // 要素がコマンドの場合

      // コマンド引数処理中ならば、処理中のコマンドをリストに追加
      if (0 != curDef)
      {
        // コマンド引数のチェック
        if (curDef->commandArgType != None)
        {
          if (currentArgs.size() < 1)
            BOOST_THROW_EXCEPTION(RuntimeException(SZT("Missing command argument")));
          if (curDef->commandArgType == Single && currentArgs.size() > 1)
            BOOST_THROW_EXCEPTION(RuntimeException(SZT("Too many command argument")));
        }
        else
        {
          if (0 != currentArgs.size())
            BOOST_THROW_EXCEPTION(RuntimeException(SZT("Unexpected command argument")));
        }

        commands.push_back(Command(curDef->commandType, currentArgs));
      }

      // 新規コマンドの処理
      curDef = def;
      currentArgs.clear();
    }
    else
    {
      if (0 == curDef)
        curDef = &comDefs[0]; // コマンド指定がない場合は展開コマンドがデフォルトとされる。

      // 要素がコマンド引数の場合
      currentArgs.push_back(*it);
    }
  }

  if (0 != curDef)
    commands.push_back(Command(curDef->commandType, currentArgs));

  return commands;
}
